function generateSpotsforDist(param, align)
% Generates spots data structure from analyzed SuperSegger folder.
% Set matlab folder to desired folder and script will collect data 
% from all xy. spots.mat containing 
% the results is saved to the same folder

% assumes spots spots in fluo1 and fluo2 channels. Fluo3 can be spots or
% just random signal

% param == 0 -> fluo3 channel with spots
% param == 1 -> no fluo3 channel - omit this field
% param == 2 -> fluo3 channel treated as area, saves image of fluo3 channel instead of spot coordinates

maxImage = 1200;        % size of the image
distEdge = 20;          % minimum distance from cell center to image edge
spotsThres = 2.5;       % score from supersegger to exclude low intensity spots

% filtering parameters to avoid clumps and not cells
minArea = 100;        
maxArea = 300;        
minwidth = 3;         
maxwidth = 9;    

if nargin == 0
    fluo3Exist = 1;
    fluo3Area = 0;
    param = 0;
elseif nargin > 0
    if param == 1
        fluo3Exist = 0;
        fluo3Area = 0;
    elseif param == 2
        fluo3Exist = 1;
        fluo3Area = 1;
    elseif param == 0
        fluo3Exist = 1;
        fluo3Area = 0;
    end   
end
if nargin < 2
    align = [];
end

%% Collect data from all XYs into a spots array
spots = struct;
spots.fluo1 = [];
spots.fluo2 = [];
spots.fluo3 = [];

n = 0;
allfolders = dir('xy*');
for jj = 1:length(allfolders)   % XY
    folder = allfolders(jj).name;
    clist = load([folder '\' 'clist.mat']);
    
    mask1 = clist.data(:,14) >= minArea & clist.data(:,14) <= maxArea;  % cell area min/max
    mask2 = clist.data(:,12) >= minwidth;	% min width
    mask3 = clist.data(:,12) <= maxwidth;	% max width
    
    mask = mask1 & mask2 & mask3;
    indeces = find(mask == 1);
    
    % find cell.mat files and order by number
    allcells = what([folder '\' 'cell' '\']);
    names = cell(1,length(allcells.mat));
    for ll = 1:length(allcells.mat)
         splitted = strsplit(allcells.mat{ll},'ll');
         names{ll} = splitted{2};
    end
    [~, I] = sort(names); 
    names = allcells.mat(I);

    for ii = 1:length(indeces)  % cells
        cellname = ['' names{indeces(ii)}];
        currentCell = load([folder '\' 'cell' '\' cellname]);
        
        % choose only cells from first frame (if time-lapse)
        if currentCell.birth == 1
            
            props=regionprops(currentCell.CellA{1}.mask,'Area', ...
            'MajorAxisLength','MinorAxisLength','Centroid');
        
            X = props.Centroid(1)+currentCell.CellA{1}.r_offset(1);
            Y = props.Centroid(2)+currentCell.CellA{1}.r_offset(2);

            % exclude cells too close to border and darker lower areas
            if X > (distEdge-1) && X < (maxImage-distEdge-1) && ...
                Y > (distEdge-1) && Y < (maxImage-distEdge-1)
            
                n = n + 1;

                spots(n).ID = currentCell.ID;
                spots(n).XY = jj;
                e1 = currentCell.CellA{1}.coord.e1;
                e2 = currentCell.CellA{1}.coord.e2;

                % locus 1
                if ~isempty(align)
                    r = [currentCell.CellA{1}.locus1(:).r];
                    r(1:2:(end-1)) = r(1:2:(end-1))+align{1}(4);
                    r(2:2:end) = r(2:2:end)+align{1}(3);
                    spots(n).fluo1.coord = r;
                    spots(n).fluo1.l = (([r(1:2:(end-1))' r(2:2:end)']-currentCell.CellA{1}.coord.rcm)*e1)';
                    spots(n).fluo1.d = (([r(1:2:(end-1))' r(2:2:end)']-currentCell.CellA{1}.coord.rcm)*e2)';
                else
                    spots(n).fluo1.coord = [currentCell.CellA{1}.locus1(:).r]; % [x y x y...]
                    spots(n).fluo1.l = [currentCell.CellA{1}.locus1(:).longaxis];
                    spots(n).fluo1.d = [currentCell.CellA{1}.locus1(:).shortaxis];
                end
                spots(n).fluo1.score = [currentCell.CellA{1}.locus1(:).score];
                spots(n).fluo1.sigma = [currentCell.CellA{1}.locus1(:).fitSigma];
                % filtering bad spots
                mask1 = spots(n).fluo1.score >= spotsThres;
                if ~isempty(mask1)
                    spots(n).fluo1.l = spots(n).fluo1.l(mask1);
                    spots(n).fluo1.d = spots(n).fluo1.d(mask1);
                    spots(n).fluo1.score = spots(n).fluo1.score(mask1);
                    spots(n).fluo1.sigma = spots(n).fluo1.sigma(mask1);
                    spots(n).fluo1.coord = spots(n).fluo1.coord(repelem(mask1,2));
                end

                % locus 2
                if isfield(currentCell.CellA{1},'locus2')
                    if ~isempty(align)
                        r = [currentCell.CellA{1}.locus2(:).r];
                        r(1:2:(end-1)) = r(1:2:(end-1))+align{2}(4);
                        r(2:2:end) = r(2:2:end)+align{2}(3);
                        spots(n).fluo2.coord = r;
                        spots(n).fluo2.l = (([r(1:2:(end-1))' r(2:2:end)']-currentCell.CellA{1}.coord.rcm)*e1)';
                        spots(n).fluo2.d = (([r(1:2:(end-1))' r(2:2:end)']-currentCell.CellA{1}.coord.rcm)*e2)';
                    else
                        spots(n).fluo2.coord = [currentCell.CellA{1}.locus2(:).r]; % [x y x y...]
                        spots(n).fluo2.l = [currentCell.CellA{1}.locus2(:).longaxis];
                        spots(n).fluo2.d = [currentCell.CellA{1}.locus2(:).shortaxis];
                    end
                    spots(n).fluo2.score = [currentCell.CellA{1}.locus2(:).score];
                    spots(n).fluo2.sigma = [currentCell.CellA{1}.locus2(:).fitSigma];
                    % filtering bad spots
                    mask2 = spots(n).fluo2.score >= spotsThres;
                    if ~isempty(mask2)
                        spots(n).fluo2.l = spots(n).fluo2.l(mask2);
                        spots(n).fluo2.d = spots(n).fluo2.d(mask2);
                        spots(n).fluo2.score = spots(n).fluo2.score(mask2);
                        spots(n).fluo2.sigma = spots(n).fluo2.sigma(mask2);
                        spots(n).fluo2.coord = spots(n).fluo2.coord(repelem(mask2,2));
                    end
                end

                % locus 3
                if fluo3Exist == 1  % if fluo3 exist
                    % fluo3 data from spot detection
                    if fluo3Area == 0   % 
                         if ~isempty(align)
                            r = [currentCell.CellA{1}.locus3(:).r];
                            r(1:2:(end-1)) = r(1:2:(end-1))+align{3}(4);
                            r(2:2:end) = r(2:2:end)+align{3}(3);
                            spots(n).fluo3.coord = r;
                            spots(n).fluo3.l = (([r(1:2:(end-1))' r(2:2:end)']-currentCell.CellA{1}.coord.rcm)*e1)';
                            spots(n).fluo3.d = (([r(1:2:(end-1))' r(2:2:end)']-currentCell.CellA{1}.coord.rcm)*e2)';
                        else
                            spots(n).fluo3.coord = [currentCell.CellA{1}.locus3(:).r]; % [x y x y...]
                            spots(n).fluo3.l = [currentCell.CellA{1}.locus3(:).longaxis];
                            spots(n).fluo3.d = [currentCell.CellA{1}.locus3(:).shortaxis];
                        end
                        spots(n).fluo3.score = [currentCell.CellA{1}.locus3(:).score];
                        spots(n).fluo3.sigma = [currentCell.CellA{1}.locus3(:).fitSigma];
                        spots(n).fluo3.int = [currentCell.CellA{1}.locus3(:).intensity];
                        % filtering bad spots
                        mask3 = spots(n).fluo3.score >= spotsThres;
                        if ~isempty(mask3)
                            spots(n).fluo3.l = spots(n).fluo3.l(mask3);
                            spots(n).fluo3.d = spots(n).fluo3.d(mask3);
                            spots(n).fluo3.score = spots(n).fluo3.score(mask3);
                            spots(n).fluo3.sigma = spots(n).fluo3.sigma(mask3);
                            spots(n).fluo3.coord = spots(n).fluo3.coord(repelem(mask3,2));
                            spots(n).fluo3.int = spots(n).fluo3.int(mask3);
                        end
                    % fluo3 data from area detection
                    elseif fluo3Area == 1
                        % analysis of fluo3 area
                        I = currentCell.CellA{1}.fluor3;
                        offSet = currentCell.CellA{1}.r_offset-1;
                        
                        % adjust cell mask with align using convexhull
                        props=regionprops(currentCell.CellA{1}.mask,'Area','Orientation', ...
                            'PixelIdxList','MajorAxisLength','MinorAxisLength','Centroid','ConvexHull');
                        
                        props.ConvexHull(:,1) = props.ConvexHull(:,1)-align{3}(4);
                        props.ConvexHull(:,2) = props.ConvexHull(:,2)-align{3}(3);
                        [yArea, xArea] = find(I);
                        in = inpolygon(xArea,yArea,props.ConvexHull(:,1),props.ConvexHull(:,2));
                        
                        % detect 5% brightest pixels from cell area
                        Ilin = I(:);
                        [f,x,~,~] = ecdf(double(Ilin(in)));
                        thresh = min(x(find(f > 0.95)));
                        % mask of px in cell area and above threshold
                        maskBright = Ilin > thresh;
                        mask = maskBright & in;

                        % add offset coordinates
                        xMask = xArea(mask) + offSet(1);
                        yMask = yArea(mask) + offSet(2);
                        r = [xMask yMask];
                        
                        if ~isempty(align)
                            r(:,1) = r(:,1)+align{3}(4);
                            r(:,2) = r(:,2)+align{3}(3);             
                        end

                        % normalized to cell coordinates
                        longaxis = (r-currentCell.CellA{1}.coord.rcm)*currentCell.CellA{1}.coord.e1;
                        shortaxis = (r-currentCell.CellA{1}.coord.rcm)*currentCell.CellA{1}.coord.e2;

                        % fluo3A coordinates are correctly aligned
                        spots(n).fluo3A.coord = r;
                        spots(n).fluo3A.l = longaxis;
                        spots(n).fluo3A.d = shortaxis;
                        spots(n).fluo3A.Int = I(mask);
                        
                        % NOTE the raw image is not aligned, 
                        % pixels selected by shifted convexhull are correct 
                        % but coordinates are still not adjusted
                        
                        % create mask from in polygon coordiates
                        maskCell = zeros(size(I,1),size(I,2));
                        maskCell(in) = 1; 
                        spots(n).fluo3Raw = bsxfun(@times, I, cast(maskCell, 'like', I));
                        spots(n).offset = offSet;
                    end

                    spots(n).fluo3Int = currentCell.CellA{1}.fl3.sum;
                    spots(n).cellFluo3Bg = currentCell.CellA{1}.fl3.bg;
                    % subtract bg fluorescence from total intensity
                    spots(n).fluo3IntBgCorr = currentCell.CellA{1}.fl3.sum-currentCell.CellA{1}.fl3.bg.*currentCell.CellA{1}.coord.A;
                end

                % cell length and width - from regionprops
                props=regionprops(currentCell.CellA{1}.mask,'Area','Orientation', ...
                    'PixelIdxList','MajorAxisLength','MinorAxisLength','Centroid','ConvexHull');
                offSet = currentCell.CellA{1}.r_offset-1; % [x y] -1 to correct for position
                spots(n).X = currentCell.CellA{1}.coord.r_center(1);
                spots(n).Y = currentCell.CellA{1}.coord.r_center(2);
                spots(n).length = props.MajorAxisLength;
                spots(n).width = props.MinorAxisLength;
                spots(n).convexHull = props.ConvexHull + offSet;  % add offset to coord  
            end
        end
    end
end

%% Colocalization distances, both intra and inter-channels between spots
% adds to spots array

for ii = 1:length(spots)
    spots(ii).coloc = [];
	for jj = 1:length(spots(ii).fluo1)
        %% calculate absolute distances
        locs1 = spots(ii).fluo1(jj).coord(:)';
        if ~isempty(locs1)  %if spots
            X1 = [locs1(1:2:end);locs1(2:2:end)]';
            X1ind = [1 size(X1,1)];
        else
            X1 = [];
            X1ind = [0 0];
        end
        
        if ~isempty(spots(ii).fluo2)
            locs2 = spots(ii).fluo2(jj).coord(:)';
            if ~isempty(locs2)  % if spots
                X2 = [locs2(1:2:end);locs2(2:2:end)]';
                X2ind = [X1ind(2)+1 X1ind(2)+size(X2,1)];
            else
                X2 = [];
                X2ind = X1ind;
            end
        else
            X2 = [];
            X2ind = X1ind;
        end
        
        if ~isempty(spots(ii).fluo3)
            locs3 = spots(ii).fluo3(jj).coord(:)';
            if ~isempty(locs3)  % if spots
                X3 = [locs3(1:2:end);locs3(2:2:end)]';
                X3ind = [X2ind(2)+1 X2ind(2)+size(X3,1)];
            else
                X3 = [];
                X3ind = X2ind;
            end
        else
            X3 = [];
            X3ind = X2ind;
        end
        
        % combine into matrix and calculate all distances
        X = [X1; X2; X3];
        d = pdist(X,'euclidean');   
        Z = squareform(d);  %distances between elements in square form

        % IFDs if at least 2 data points in each channel
        if ~isempty(X1) && size(X1,1) >= 2
            spots(ii).coloc(jj).IFD1 = Z((X1ind(1)+1):X1ind(2),X1ind(1));
            spots(ii).coloc(jj).IFD1_min = min(spots(ii).coloc(jj).IFD1);
        else
            spots(ii).coloc(jj).IFD1 = NaN;
            spots(ii).coloc(jj).IFD1_min = NaN;
        end
        if ~isempty(X2) && size(X2,1) >= 2
            spots(ii).coloc(jj).IFD2 = Z((X2ind(1)+1):X2ind(2),X2ind(1));
            spots(ii).coloc(jj).IFD2_min = min(spots(ii).coloc(jj).IFD2);
        else
            spots(ii).coloc(jj).IFD2 = NaN;
            spots(ii).coloc(jj).IFD2_min = NaN;
        end
        if ~isempty(X3) && size(X3,1) >= 2
            spots(ii).coloc(jj).IFD3 = Z((X3ind(1)+1):X3ind(2),X3ind(1));
            spots(ii).coloc(jj).IFD3_min = min(spots(ii).coloc(jj).IFD3);
        else
            spots(ii).coloc(jj).IFD3 = NaN;
            spots(ii).coloc(jj).IFD3_min = NaN;
        end
        
        % inter-channel distances for each channe;
        if ~isempty(X1) && ~isempty(X2)
            dists1_2 = Z(X1ind(1):X1ind(2),X2ind(1):X2ind(2));
            spots(ii).coloc(jj).dists1_2 = dists1_2(:);
            spots(ii).coloc(jj).dists1_2_min = min(spots(ii).coloc(jj).dists1_2);
        else
            spots(ii).coloc(jj).dists1_2 = NaN;
            spots(ii).coloc(jj).dists1_2_min = NaN;
        end
        if ~isempty(X1) && ~isempty(X3)
            dists1_3 = Z(X1ind(1):X1ind(2),X3ind(1):X3ind(2));
            spots(ii).coloc(jj).dists1_3 = dists1_3(:);
            spots(ii).coloc(jj).dists1_3_min = min(spots(ii).coloc(jj).dists1_3);
        else
            spots(ii).coloc(jj).dists1_3 = NaN;
            spots(ii).coloc(jj).dists1_3_min = NaN;
        end
        if ~isempty(X2) && ~isempty(X3)
            dists2_3 = Z(X2ind(1):X2ind(2),X3ind(1):X3ind(2));
            spots(ii).coloc(jj).dists2_3 = dists2_3(:);
            spots(ii).coloc(jj).dists2_3_min = min(spots(ii).coloc(jj).dists2_3);
        else
            spots(ii).coloc(jj).dists2_3 = NaN;
            spots(ii).coloc(jj).dists2_3_min = NaN;
        end
	end
end

folder = pwd;
newStr = split(folder,'\');

% saves spots data structure with folder name
if param == 0
    name = char(strcat(newStr(end),'_pop_spots.mat'));
    save(name,'spots','align')
elseif param == 2
    name = char(strcat(newStr(end),'_pop_bright.mat'));
    save(name,'spots','align')
else
    name = char(strcat(newStr(end),'_pop.mat'));
    save(name,'spots')
end

end

